<?php

namespace Tests;

use Illuminate\Foundation\Testing\TestCase as BaseTestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use App\Http\Middleware\CheckRole;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Config;
use Spatie\Permission\PermissionRegistrar;
use Tests\CreatesApplication;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Artisan;

abstract class TestCase extends BaseTestCase
{
    use CreatesApplication;
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // SQLite için yapılandırma
        if (DB::connection()->getDriverName() === 'sqlite') {
            DB::statement('PRAGMA journal_mode = MEMORY');
            DB::statement('PRAGMA synchronous = OFF');
            DB::statement('PRAGMA foreign_keys = ON');
            DB::statement('PRAGMA temp_store = MEMORY');
            DB::statement('PRAGMA cache_size = 10000');
            DB::statement('PRAGMA busy_timeout = 5000');
        }

        // API guard'ı ayarla
        Config::set('auth.defaults.guard', 'api');

        // Cache'i devre dışı bırak
        Config::set('cache.default', 'array');
        Config::set('cache.stores.array', [
            'driver' => 'array'
        ]);

        // Spatie Permission cache'ini devre dışı bırak
        Config::set('permission.cache.expiration_time', 0);
        Config::set('permission.cache.store', null);
        Config::set('permission.enable_cache', false);

        // Spatie Permission cache'ini temizle
        app()[PermissionRegistrar::class]->forgetCachedPermissions();

        // Register middleware
        $this->app->singleton(CheckRole::class);
    }

    protected function tearDown(): void
    {
        if (DB::connection()->getDriverName() === 'sqlite') {
            DB::statement('PRAGMA journal_mode = DELETE');
            DB::statement('PRAGMA synchronous = FULL');
        }

        parent::tearDown();
    }

    protected function refreshDatabase()
    {
        // Tüm tabloları temizle
        Schema::disableForeignKeyConstraints();
        $tables = DB::select("SELECT name FROM sqlite_master WHERE type='table' AND name NOT LIKE 'sqlite_%'");
        foreach ($tables as $table) {
            DB::statement("DROP TABLE IF EXISTS {$table->name}");
        }
        Schema::enableForeignKeyConstraints();

        // Temel migrasyonları sırayla çalıştır
        $migrations = [
            'database/migrations/0001_01_01_000000_create_users_table.php',
            'database/migrations/0001_01_01_000001_create_cache_table.php',
            'database/migrations/0001_01_01_000002_create_jobs_table.php',
            'database/migrations/2025_04_22_115714_create_permission_tables.php'
        ];

        foreach ($migrations as $migration) {
            Artisan::call('migrate', [
                '--path' => $migration,
                '--force' => true
            ]);
        }

        // Diğer migrasyonları çalıştır
        Artisan::call('migrate', [
            '--force' => true
        ]);
    }

    protected function defineDatabaseMigrations()
    {
        $this->loadMigrationsFrom(database_path('migrations'));
    }
}
