<?php

namespace Tests\Feature;

use App\Models\ServiceRecord;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ServiceRecordSearchTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $technician;
    protected $serviceRecords = [];

    protected function setUp(): void
    {
        parent::setUp();

        // Admin kullanıcısı oluştur
        $this->admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin' . uniqid() . '@example.com'
        ]);

        // Teknik servis personeli oluştur
        $this->technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician' . uniqid() . '@example.com'
        ]);

        // Test verilerini oluştur
        $this->serviceRecords = [
            'pending' => ServiceRecord::factory()->create([
                'tracking_number' => '202403210001',
                'customer_name' => 'Ahmet Yılmaz',
                'customer_phone' => '5551234567',
                'device_model' => 'iPhone 12',
                'device_serial_number' => 'SN12345678',
                'status' => ServiceRecord::STATUS_PENDING,
                'created_at' => now()->subDays(2)
            ]),
            'in_progress' => ServiceRecord::factory()->create([
                'tracking_number' => '202403210002',
                'customer_name' => 'Mehmet Demir',
                'customer_phone' => '5559876543',
                'device_model' => 'Samsung S21',
                'device_serial_number' => 'SN87654321',
                'status' => ServiceRecord::STATUS_IN_PROGRESS,
                'technician_id' => $this->technician->id,
                'created_at' => now()->subDays(1)
            ]),
            'completed' => ServiceRecord::factory()->create([
                'tracking_number' => '202403210003',
                'customer_name' => 'Ayşe Kaya',
                'customer_phone' => '5554567890',
                'device_model' => 'Huawei P40',
                'device_serial_number' => 'SN98765432',
                'status' => ServiceRecord::STATUS_COMPLETED,
                'technician_id' => $this->technician->id,
                'created_at' => now()
            ])
        ];
    }

    public function test_can_search_by_tracking_number()
    {
        $this->actingAs($this->admin);

        $response = $this->get(route('service-records.index', [
            'search' => '202403210001'
        ]));

        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) {
            return $records->count() === 1 &&
                   $records->first()->id === $this->serviceRecords['pending']->id;
        });
    }

    public function test_can_search_by_customer_name()
    {
        $this->actingAs($this->admin);

        $response = $this->get(route('service-records.index', [
            'search' => 'Mehmet'
        ]));

        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) {
            return $records->count() === 1 &&
                   $records->first()->id === $this->serviceRecords['in_progress']->id;
        });
    }

    public function test_can_search_by_device_model()
    {
        $this->actingAs($this->admin);

        $response = $this->get(route('service-records.index', [
            'search' => 'iPhone'
        ]));

        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) {
            return $records->count() === 1 &&
                   $records->first()->id === $this->serviceRecords['pending']->id;
        });
    }

    public function test_can_filter_by_status()
    {
        $this->actingAs($this->admin);

        $response = $this->get(route('service-records.index', [
            'status' => ServiceRecord::STATUS_IN_PROGRESS
        ]));

        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) {
            return $records->count() === 1 &&
                   $records->first()->id === $this->serviceRecords['in_progress']->id;
        });
    }

    public function test_can_filter_by_technician()
    {
        $this->actingAs($this->admin);

        $response = $this->get(route('service-records.index', [
            'technician_id' => $this->technician->id
        ]));

        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) {
            return $records->count() === 2 &&
                   $records->pluck('id')->contains($this->serviceRecords['in_progress']->id) &&
                   $records->pluck('id')->contains($this->serviceRecords['completed']->id);
        });
    }

    public function test_can_filter_by_date_range()
    {
        $this->actingAs($this->admin);

        $response = $this->get(route('service-records.index', [
            'date_from' => now()->subDays(1)->format('Y-m-d'),
            'date_to' => now()->format('Y-m-d')
        ]));

        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) {
            return $records->count() === 2 &&
                   $records->pluck('id')->contains($this->serviceRecords['in_progress']->id) &&
                   $records->pluck('id')->contains($this->serviceRecords['completed']->id);
        });
    }

    public function test_can_sort_by_created_at()
    {
        $this->actingAs($this->admin);

        $response = $this->get(route('service-records.index', [
            'sort' => 'created_at',
            'direction' => 'asc'
        ]));

        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) {
            return $records->count() === 3 &&
                   $records->first()->id === $this->serviceRecords['pending']->id &&
                   $records->last()->id === $this->serviceRecords['completed']->id;
        });
    }

    public function test_can_combine_search_and_filters()
    {
        $this->actingAs($this->admin);

        $response = $this->get(route('service-records.index', [
            'search' => 'iPhone',
            'status' => ServiceRecord::STATUS_PENDING,
            'date_from' => now()->subDays(3)->format('Y-m-d'),
            'date_to' => now()->format('Y-m-d')
        ]));

        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) {
            return $records->count() === 1 &&
                   $records->first()->id === $this->serviceRecords['pending']->id;
        });
    }
} 