<?php

namespace Tests\Feature;

use App\Models\ServiceRecord;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ServiceRecordFunctionalTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $serviceManager;
    protected $serviceStaff;
    protected $technician;
    protected $customer;

    protected function setUp(): void
    {
        parent::setUp();

        // Admin kullanıcısı oluştur
        $this->admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin@example.com'
        ]);

        // Servis yöneticisi oluştur
        $this->serviceManager = User::factory()->create([
            'role' => 'service_manager',
            'email' => 'manager@example.com'
        ]);

        // Servis personeli oluştur
        $this->serviceStaff = User::factory()->create([
            'role' => 'service_staff',
            'email' => 'staff@example.com'
        ]);

        // Teknik servis personeli oluştur
        $this->technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician@example.com'
        ]);

        // Müşteri oluştur
        $this->customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer@example.com'
        ]);
    }

    public function test_can_create_service_record()
    {
        $this->actingAs($this->admin);

        $response = $this->post(route('service-records.store'), [
            'customer_name' => 'Test Customer',
            'customer_phone' => '5551234567',
            'customer_email' => 'test@example.com',
            'customer_address' => 'Test Address',
            'device_brand' => 'Test Brand',
            'device_model' => 'Test Model',
            'device_serial_number' => 'SN123456',
            'fault_description' => 'Test Description',
            'status' => ServiceRecord::STATUS_PENDING
        ]);

        $response->assertRedirect(route('service-records.show', ServiceRecord::latest()->first()));
        $this->assertDatabaseHas('service_records', [
            'customer_name' => 'Test Customer',
            'device_serial_number' => 'SN123456',
            'status' => ServiceRecord::STATUS_PENDING
        ]);
    }

    public function test_can_update_service_record()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING
        ]);

        $response = $this->put(route('service-records.update', $serviceRecord), [
            'customer_name' => 'Updated Customer',
            'customer_phone' => '5559876543',
            'customer_email' => 'updated@example.com',
            'customer_address' => 'Updated Address',
            'device_brand' => 'Updated Brand',
            'device_model' => 'Updated Model',
            'device_serial_number' => 'SN654321',
            'fault_description' => 'Updated Description',
            'status' => ServiceRecord::STATUS_IN_PROGRESS
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
        $this->assertDatabaseHas('service_records', [
            'id' => $serviceRecord->id,
            'customer_name' => 'Updated Customer',
            'device_serial_number' => 'SN654321',
            'status' => ServiceRecord::STATUS_IN_PROGRESS
        ]);
    }

    public function test_can_change_service_record_status()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING
        ]);

        $response = $this->post(route('service-records.change-status', $serviceRecord), [
            'new_status' => ServiceRecord::STATUS_IN_PROGRESS,
            'description' => 'Status changed to in progress',
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
        $this->assertDatabaseHas('service_records', [
            'id' => $serviceRecord->id,
            'status' => ServiceRecord::STATUS_IN_PROGRESS
        ]);
        $this->assertDatabaseHas('service_status_changes', [
            'service_record_id' => $serviceRecord->id,
            'old_status' => ServiceRecord::STATUS_PENDING,
            'new_status' => ServiceRecord::STATUS_IN_PROGRESS,
            'description' => 'Status changed to in progress'
        ]);
    }

    public function test_can_assign_technician()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'technician_id' => null
        ]);

        $response = $this->post(route('service-records.assign-technician', $serviceRecord), [
            'technician_id' => $this->technician->id,
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
        $this->assertDatabaseHas('service_records', [
            'id' => $serviceRecord->id,
            'technician_id' => $this->technician->id
        ]);
    }

    public function test_can_reassign_technician()
    {
        $this->actingAs($this->admin);

        $oldTechnician = User::factory()->create([
            'role' => 'technician',
            'email' => 'old_technician@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_IN_PROGRESS,
            'technician_id' => $oldTechnician->id
        ]);

        $response = $this->post(route('service-records.assign-technician', $serviceRecord), [
            'technician_id' => $this->technician->id,
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
        $this->assertDatabaseHas('service_records', [
            'id' => $serviceRecord->id,
            'technician_id' => $this->technician->id
        ]);
    }

    public function test_can_complete_service_record()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_IN_PROGRESS,
            'technician_id' => $this->technician->id
        ]);

        $response = $this->post(route('service-records.change-status', $serviceRecord), [
            'new_status' => ServiceRecord::STATUS_COMPLETED,
            'description' => 'Service completed successfully',
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
        $this->assertDatabaseHas('service_records', [
            'id' => $serviceRecord->id,
            'status' => ServiceRecord::STATUS_COMPLETED
        ]);
    }

    public function test_can_cancel_service_record()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING
        ]);

        $response = $this->post(route('service-records.change-status', $serviceRecord), [
            'new_status' => ServiceRecord::STATUS_CANCELLED,
            'description' => 'Service cancelled by customer request',
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
        $this->assertDatabaseHas('service_records', [
            'id' => $serviceRecord->id,
            'status' => ServiceRecord::STATUS_CANCELLED
        ]);
    }

    public function test_can_search_service_records()
    {
        $this->actingAs($this->admin);

        // Test kayıtları oluştur
        $record1 = ServiceRecord::factory()->create([
            'customer_name' => 'John Doe',
            'device_serial_number' => 'SN123456'
        ]);

        $record2 = ServiceRecord::factory()->create([
            'customer_name' => 'Jane Smith',
            'device_serial_number' => 'SN789012'
        ]);

        // Müşteri adına göre arama
        $response = $this->get(route('service-records.index', ['search' => 'John']));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) use ($record1) {
            return $records->contains($record1) && !$records->contains('customer_name', 'Jane');
        });

        // Seri numarasına göre arama
        $response = $this->get(route('service-records.index', ['search' => 'SN789']));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) use ($record2) {
            return $records->contains($record2) && !$records->contains('device_serial_number', 'SN123456');
        });
    }

    public function test_can_filter_service_records()
    {
        $this->actingAs($this->admin);

        // Test kayıtları oluştur
        $pendingRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING
        ]);

        $inProgressRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_IN_PROGRESS,
            'technician_id' => $this->technician->id
        ]);

        // Duruma göre filtreleme
        $response = $this->get(route('service-records.index', ['status' => ServiceRecord::STATUS_PENDING]));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) use ($pendingRecord) {
            return $records->contains($pendingRecord) && !$records->contains('status', ServiceRecord::STATUS_IN_PROGRESS);
        });

        // Teknik servise göre filtreleme
        $response = $this->get(route('service-records.index', ['technician_id' => $this->technician->id]));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) use ($inProgressRecord) {
            return $records->contains($inProgressRecord) && !$records->contains('technician_id', null);
        });
    }

    public function test_can_sort_service_records()
    {
        $this->actingAs($this->admin);

        // Test kayıtları oluştur
        $record1 = ServiceRecord::factory()->create([
            'created_at' => now()->subDays(2)
        ]);

        $record2 = ServiceRecord::factory()->create([
            'created_at' => now()->subDays(1)
        ]);

        // Tarihe göre sıralama (en yeni)
        $response = $this->get(route('service-records.index', [
            'sort' => 'created_at',
            'direction' => 'desc'
        ]));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) use ($record1, $record2) {
            return $records->first()->id === $record2->id && $records->last()->id === $record1->id;
        });

        // Tarihe göre sıralama (en eski)
        $response = $this->get(route('service-records.index', [
            'sort' => 'created_at',
            'direction' => 'asc'
        ]));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords', function ($records) use ($record1, $record2) {
            return $records->first()->id === $record1->id && $records->last()->id === $record2->id;
        });
    }
} 