<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, Notifiable, SoftDeletes, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'phone',
        'address',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
    ];

    /**
     * Check if the user is an admin.
     *
     * @return bool
     */
    public function isAdmin(): bool
    {
        return $this->hasRole('admin');
    }

    /**
     * Check if the user is a service manager.
     *
     * @return bool
     */
    public function isServiceManager(): bool
    {
        return $this->hasRole('service_manager');
    }

    /**
     * Check if the user is a service staff member.
     *
     * @return bool
     */
    public function isServiceStaff(): bool
    {
        return $this->hasRole('service_staff');
    }

    /**
     * Check if the user is a technician.
     *
     * @return bool
     */
    public function isTechnician(): bool
    {
        return $this->hasRole('technician');
    }

    /**
     * Check if the user is a customer.
     *
     * @return bool
     */
    public function isCustomer(): bool
    {
        return $this->hasRole('customer');
    }

    /**
     * Get the assigned service records for this user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function assignedServices()
    {
        return $this->hasMany(ServiceRecord::class, 'technician_id');
    }

    /**
     * Get the status changes made by this user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function statusChanges()
    {
        return $this->hasMany(ServiceStatusChange::class, 'changed_by_id');
    }

    /**
     * Scope query to only include service staff users.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeServiceStaff($query)
    {
        return $query->role('service_staff');
    }

    /**
     * Scope query to only include technicians.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeTechnicians($query)
    {
        return $query->role('technician');
    }

    /**
     * Scope query to only include active technicians.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActiveTechnicians($query)
    {
        return $query->technicians()->whereNull('deleted_at');
    }

    /**
     * Scope query to only include technicians in a specific region.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $region
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeTechniciansInRegion($query, string $region)
    {
        return $query->technicians()->where('region', $region);
    }

    /**
     * Scope query to only include available technicians.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $maxAssignedServices
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAvailableTechnicians($query, int $maxAssignedServices = 5)
    {
        return $query->technicians()
            ->whereNull('deleted_at')
            ->whereDoesntHave('assignedServices', function ($q) use ($maxAssignedServices) {
                $q->whereIn('status', [
                    ServiceRecord::STATUS_IN_PROGRESS,
                    ServiceRecord::STATUS_WAITING_FOR_PARTS
                ]);
            }, '>=', $maxAssignedServices);
    }

    /**
     * Get the count of assigned service records.
     *
     * @return int
     */
    public function getAssignedServicesCountAttribute(): int
    {
        return $this->assignedServices()->count();
    }

    /**
     * Get the count of active service records.
     *
     * @return int
     */
    public function getActiveServicesCountAttribute(): int
    {
        return $this->activeServices()->count();
    }

    /**
     * Get the active (not completed) service records.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function activeServices()
    {
        return $this->assignedServices()
            ->whereNotIn('status', [
                ServiceRecord::STATUS_COMPLETED,
                ServiceRecord::STATUS_CANCELLED
            ]);
    }

    /**
     * Get all roles for this user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function roles(): BelongsToMany
    {
        return $this->morphToMany(
            config('permission.models.role'),
            'model',
            config('permission.table_names.model_has_roles'),
            config('permission.column_names.model_morph_key'),
            'role_id'
        );
    }

    /**
     * Check if the user has any of the given roles.
     *
     * @param array|string[] $roles
     * @return bool
     */
    public function hasAnyRole(array $roles): bool
    {
        return $this->roles()->whereIn('name', $roles)->exists();
    }
}
