<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ServiceStatus extends Model
{
    use HasFactory;

    protected $fillable = [
        'service_record_id',
        'status',
        'description',
        'created_by',
        'notify_customer',
        'notification_sent_at'
    ];

    protected $casts = [
        'notify_customer' => 'boolean',
        'notification_sent_at' => 'datetime'
    ];

    // Status Constants
    const STATUS_RECEIVED = 'received';
    const STATUS_DIAGNOSING = 'diagnosing';
    const STATUS_WAITING_APPROVAL = 'waiting_approval';
    const STATUS_APPROVED = 'approved';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_WAITING_PARTS = 'waiting_parts';
    const STATUS_COMPLETED = 'completed';
    const STATUS_READY_FOR_PICKUP = 'ready_for_pickup';
    const STATUS_DELIVERED = 'delivered';
    const STATUS_CANCELLED = 'cancelled';

    public static function getStatusList()
    {
        return [
            self::STATUS_RECEIVED => 'Teslim Alındı',
            self::STATUS_DIAGNOSING => 'Arıza Tespiti',
            self::STATUS_WAITING_APPROVAL => 'Onay Bekliyor',
            self::STATUS_APPROVED => 'Onaylandı',
            self::STATUS_IN_PROGRESS => 'İşlemde',
            self::STATUS_WAITING_PARTS => 'Parça Bekliyor',
            self::STATUS_COMPLETED => 'Tamamlandı',
            self::STATUS_READY_FOR_PICKUP => 'Teslime Hazır',
            self::STATUS_DELIVERED => 'Teslim Edildi',
            self::STATUS_CANCELLED => 'İptal Edildi'
        ];
    }

    // Relationships
    public function serviceRecord()
    {
        return $this->belongsTo(ServiceRecord::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scopes
    public function scopeNotified($query)
    {
        return $query->whereNotNull('notification_sent_at');
    }

    public function scopePendingNotification($query)
    {
        return $query->where('notify_customer', true)
                    ->whereNull('notification_sent_at');
    }

    // Methods
    public function markNotificationSent()
    {
        $this->notification_sent_at = now();
        return $this->save();
    }

    public function getStatusTextAttribute()
    {
        return self::getStatusList()[$this->status] ?? $this->status;
    }
}
