<?php

namespace App\Http\Controllers\Technician;

use App\Http\Controllers\Controller;
use App\Models\ServiceRecord;
use App\Models\ServiceRecordNote;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TechnicianServiceRecordController extends Controller
{
    public function index(Request $request)
    {
        $technician = Auth::user();

        $query = ServiceRecord::where('technician_id', $technician->id)
            ->with(['customer', 'notes'])
            ->orderBy('priority', 'desc')
            ->orderBy('created_at', 'desc');

        // Filtreleme
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->has('priority') && $request->priority !== 'all') {
            $query->where('priority', $request->priority);
        }

        // Arama
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('tracking_number', 'like', "%{$search}%")
                    ->orWhere('brand', 'like', "%{$search}%")
                    ->orWhere('model', 'like', "%{$search}%")
                    ->orWhere('serial_number', 'like', "%{$search}%")
                    ->orWhereHas('customer', function($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            });
        }

        $serviceRecords = $query->paginate(10);

        return view('technician.service-records.index', compact('serviceRecords'));
    }

    public function show(ServiceRecord $serviceRecord)
    {
        $this->authorize('view', $serviceRecord);

        $serviceRecord->load(['customer', 'technician', 'notes' => function($query) {
            $query->orderBy('created_at', 'desc');
        }]);

        return view('technician.service-records.show', compact('serviceRecord'));
    }

    public function updateStatus(Request $request, ServiceRecord $serviceRecord)
    {
        $this->authorize('update', $serviceRecord);

        $request->validate([
            'status' => 'required|in:pending,in_progress,waiting_for_parts,completed,cancelled',
            'note' => 'required|string|max:1000'
        ]);

        DB::transaction(function() use ($request, $serviceRecord) {
            $serviceRecord->update(['status' => $request->status]);

            ServiceRecordNote::create([
                'service_record_id' => $serviceRecord->id,
                'content' => $request->note,
                'type' => 'status_change'
            ]);
        });

        return back()->with('success', 'Servis durumu güncellendi.');
    }

    public function addNote(Request $request, ServiceRecord $serviceRecord)
    {
        $this->authorize('update', $serviceRecord);

        $request->validate([
            'content' => 'required|string|max:1000',
            'type' => 'required|in:technician_note,general'
        ]);

        ServiceRecordNote::create([
            'service_record_id' => $serviceRecord->id,
            'content' => $request->content,
            'type' => $request->type
        ]);

        return back()->with('success', 'Not eklendi.');
    }
} 