<?php

namespace App\Http\Controllers\Technician;

use App\Http\Controllers\Controller;
use App\Models\ServiceRecord;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TechnicianReportController extends Controller
{
    public function index(Request $request)
    {
        // Tarih aralığını belirle
        $startDate = $request->start_date ? Carbon::parse($request->start_date) : Carbon::now()->startOfMonth();
        $endDate = $request->end_date ? Carbon::parse($request->end_date) : Carbon::now();

        // Genel istatistikler
        $stats = ServiceRecord::where('technician_id', Auth::id())
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('
                COUNT(*) as total,
                SUM(CASE WHEN status = "completed" THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = "in_progress" THEN 1 ELSE 0 END) as in_progress,
                SUM(CASE WHEN status = "waiting_for_parts" THEN 1 ELSE 0 END) as waiting_for_parts,
                SUM(CASE WHEN status = "cancelled" THEN 1 ELSE 0 END) as cancelled
            ')
            ->first()
            ->toArray();

        // Ortalama tamamlanma süresi (saat cinsinden)
        $averageCompletionTime = ServiceRecord::where('technician_id', Auth::id())
            ->where('status', 'completed')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('AVG(TIMESTAMPDIFF(HOUR, created_at, updated_at)) as avg_time')
            ->first()
            ->avg_time ?? 0;

        // Marka bazlı servis dağılımı
        $brandStats = ServiceRecord::where('technician_id', Auth::id())
            ->whereBetween('created_at', [$startDate, $endDate])
            ->select('brand', DB::raw('COUNT(*) as total'))
            ->groupBy('brand')
            ->orderBy('total', 'desc')
            ->get();

        // Günlük servis sayıları
        $dailyStats = ServiceRecord::where('technician_id', Auth::id())
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as date, COUNT(*) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get()
            ->map(function ($item) {
                return [
                    'date' => Carbon::parse($item->date)->format('d.m.Y'),
                    'total' => $item->total
                ];
            });

        return view('technician.reports.index', compact(
            'startDate',
            'endDate',
            'stats',
            'averageCompletionTime',
            'brandStats',
            'dailyStats'
        ));
    }
} 