<?php

namespace App\Http\Controllers;

use App\Models\ServiceRecord;
use App\Models\Customer;
use App\Models\Technician;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ServiceRecordStatisticsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api');
        $this->middleware('permission:view statistics');
    }

    public function statusDistribution(Request $request)
    {
        $query = ServiceRecord::query();

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                Carbon::parse($request->start_date),
                Carbon::parse($request->end_date)
            ]);
        }

        $stats = $query->groupBy('status')
            ->select('status', DB::raw('count(*) as count'))
            ->get()
            ->pluck('count', 'status')
            ->toArray();

        return response()->json([
            'data' => [
                'completed' => $stats['completed'] ?? 0,
                'in_progress' => $stats['in_progress'] ?? 0,
                'pending' => $stats['pending'] ?? 0,
                'cancelled' => $stats['cancelled'] ?? 0
            ]
        ]);
    }

    public function technicianPerformance()
    {
        $stats = ServiceRecord::join('technicians', 'service_records.technician_id', '=', 'technicians.id')
            ->select(
                'technicians.name as technician_name',
                DB::raw('COUNT(CASE WHEN service_records.status = "completed" THEN 1 END) as completed_services'),
                DB::raw('AVG(CASE WHEN service_records.status = "completed" THEN service_records.completion_time END) as average_completion_time'),
                DB::raw('0 as customer_satisfaction') // Placeholder for future implementation
            )
            ->groupBy('technicians.id', 'technicians.name')
            ->get();

        return response()->json(['data' => $stats]);
    }

    public function completionTime()
    {
        $stats = ServiceRecord::whereNotNull('completion_time')
            ->where('status', 'completed')
            ->select(
                DB::raw('AVG(completion_time) as average_completion_time'),
                DB::raw('MIN(completion_time) as min_completion_time'),
                DB::raw('MAX(completion_time) as max_completion_time')
            )
            ->first();

        return response()->json(['data' => $stats]);
    }

    public function monthly()
    {
        $stats = ServiceRecord::select(
            DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
            DB::raw('COUNT(*) as total_services'),
            DB::raw('COUNT(CASE WHEN status = "completed" THEN 1 END) as completed_services'),
            DB::raw('0 as revenue') // Placeholder for future implementation
        )
        ->groupBy('month')
        ->orderBy('month', 'desc')
        ->get();

        return response()->json(['data' => $stats]);
    }

    public function customers()
    {
        $totalCustomers = Customer::count();
        $activeCustomers = Customer::whereHas('serviceRecords', function ($query) {
            $query->where('created_at', '>=', now()->subMonths(3));
        })->count();
        $newCustomersThisMonth = Customer::where('created_at', '>=', now()->startOfMonth())->count();

        return response()->json([
            'data' => [
                'total_customers' => $totalCustomers,
                'active_customers' => $activeCustomers,
                'new_customers_this_month' => $newCustomersThisMonth,
                'customer_satisfaction' => 0 // Placeholder for future implementation
            ]
        ]);
    }
} 