<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\ServiceStatus;
use App\Models\ServiceRecord;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ServiceStatusController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, ServiceRecord $serviceRecord)
    {
        $query = $serviceRecord->statuses()->with('creator');

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $query->orderBy('created_at', 'desc');

        return response()->json([
            'success' => true,
            'data' => $query->paginate($request->input('per_page', 15))
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, ServiceRecord $serviceRecord)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|string',
            'description' => 'nullable|string',
            'notify_customer' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $status = $serviceRecord->updateStatus(
            $request->status,
            $request->description,
            auth()->id(),
            $request->boolean('notify_customer', false)
        );

        return response()->json([
            'success' => true,
            'data' => $status,
            'message' => 'Durum başarıyla eklendi.'
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(ServiceStatus $status)
    {
        return response()->json([
            'success' => true,
            'data' => $status->load(['creator', 'serviceRecord'])
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ServiceStatus $status)
    {
        $validator = Validator::make($request->all(), [
            'description' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $status->update([
            'description' => $request->description
        ]);

        return response()->json([
            'success' => true,
            'data' => $status->fresh(),
            'message' => 'Durum başarıyla güncellendi.'
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ServiceStatus $status)
    {
        $status->delete();

        return response()->json([
            'success' => true,
            'message' => 'Durum başarıyla silindi.'
        ]);
    }

    public function latest(ServiceRecord $serviceRecord)
    {
        $latestStatus = $serviceRecord->statuses()
            ->with('creator')
            ->latest()
            ->first();

        return response()->json([
            'success' => true,
            'data' => $latestStatus
        ]);
    }
}
