<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ServiceRecord;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use App\Notifications\ServiceStatusUpdated;
use App\Notifications\TechnicianAssigned;
use Carbon\Carbon;

class ServiceRecordController extends Controller
{
    public function __construct()
    {
        // Middleware'i route dosyasında tanımlıyoruz
    }

    /**
     * Display a listing of the service records.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $query = ServiceRecord::query();

        // Müşteri sadece kendi kayıtlarını görebilir
        if ($request->user()->role === 'customer') {
            $query->where('customer_email', $request->user()->email);
        }

        // Teknisyen sadece kendisine atanan kayıtları görebilir
        if ($request->user()->role === 'technician') {
            $query->where('technician_id', $request->user()->id);
        }

        // Arama
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('tracking_number', 'like', "%{$search}%")
                    ->orWhere('customer_name', 'like', "%{$search}%")
                    ->orWhere('customer_email', 'like', "%{$search}%")
                    ->orWhere('device_brand', 'like', "%{$search}%")
                    ->orWhere('device_model', 'like', "%{$search}%")
                    ->orWhere('device_serial_number', 'like', "%{$search}%");
            });
        }

        // Filtreleme
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('technician_id')) {
            $query->where('technician_id', $request->technician_id);
        }

        // Sıralama
        $sortField = $request->input('sort_field', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortField, $sortOrder);

        // Sayfalama
        $perPage = $request->input('per_page', 10);
        $records = $query->paginate($perPage);

        return response()->json([
            'data' => $records->items(),
            'meta' => [
                'current_page' => $records->currentPage(),
                'per_page' => $records->perPage(),
                'total' => $records->total()
            ]
        ]);
    }

    /**
     * Store a newly created service record.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email|max:255',
            'customer_phone' => 'required|string|max:20',
            'customer_address' => 'nullable|string|max:255',
            'device_brand' => 'required|string|max:255',
            'device_model' => 'required|string|max:255',
            'device_serial_number' => 'required|string|max:255',
            'fault_description' => 'required|string',
            'status' => 'required|in:pending,in_progress,waiting_for_parts,completed,cancelled'
        ]);

        $validated['tracking_number'] = ServiceRecord::generateTrackingNumber();
        $validated['created_by'] = $request->user()->id;

        $record = ServiceRecord::create($validated);

        return response()->json([
            'data' => $record
        ], 201);
    }

    /**
     * Display the specified service record.
     *
     * @param  \App\Models\ServiceRecord  $serviceRecord
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(ServiceRecord $serviceRecord)
    {
        // Müşteri sadece kendi kaydını görebilir
        if (request()->user()->role === 'customer' && $serviceRecord->customer_email !== request()->user()->email) {
            return response()->json(['message' => 'Bu kayda erişim yetkiniz yok.'], 403);
        }

        // Teknisyen sadece kendisine atanan kaydı görebilir
        if (request()->user()->role === 'technician' && $serviceRecord->technician_id !== request()->user()->id) {
            return response()->json(['message' => 'Bu kayda erişim yetkiniz yok.'], 403);
        }

        return response()->json([
            'data' => $serviceRecord->load(['technician', 'statusHistory', 'items', 'items.specifications', 'items.accessories'])
        ]);
    }

    /**
     * Update the specified service record.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\ServiceRecord  $serviceRecord
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, ServiceRecord $serviceRecord)
    {
        // Teknisyen sadece kendisine atanan kaydı güncelleyebilir
        if ($request->user()->role === 'technician' && $serviceRecord->technician_id !== $request->user()->id) {
            return response()->json(['message' => 'Bu kaydı güncelleme yetkiniz yok.'], 403);
        }

        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email|max:255',
            'customer_phone' => 'required|string|max:20',
            'customer_address' => 'nullable|string|max:255',
            'device_brand' => 'required|string|max:255',
            'device_model' => 'required|string|max:255',
            'device_serial_number' => 'required|string|max:255',
            'fault_description' => 'required|string',
            'status' => 'required|in:pending,in_progress,waiting_for_parts,completed,cancelled'
        ]);

        $serviceRecord->update($validated);

        return response()->json([
            'data' => $serviceRecord
        ]);
    }

    /**
     * Change the status of the specified service record.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\ServiceRecord  $serviceRecord
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeStatus(Request $request, ServiceRecord $serviceRecord)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,in_progress,waiting_for_parts,completed,cancelled',
            'notes' => 'nullable|string'
        ]);

        $oldStatus = $serviceRecord->status;
        $newStatus = $validated['status'];

        if ($oldStatus === $newStatus) {
            return response()->json(['message' => 'Durum zaten bu değerde.'], 422);
        }

        DB::transaction(function () use ($serviceRecord, $validated, $oldStatus, $newStatus, $request) {
            $serviceRecord->update(['status' => $newStatus]);

            $serviceRecord->statusHistory()->create([
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
                'changed_by' => $request->user()->id,
                'notes' => $validated['notes'] ?? null
            ]);

            // Müşteriye bildirim gönder
            if ($serviceRecord->customer_email) {
                Mail::to($serviceRecord->customer_email)->send(
                    new ServiceStatusUpdated($serviceRecord, $newStatus, $request->user())
                );
            }
        });

        return response()->json([
            'data' => $serviceRecord->fresh()
        ]);
    }

    /**
     * Assign a technician to the specified service record.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\ServiceRecord  $serviceRecord
     * @return \Illuminate\Http\JsonResponse
     */
    public function assignTechnician(Request $request, ServiceRecord $serviceRecord)
    {
        $validated = $request->validate([
            'technician_id' => 'required|exists:users,id',
            'send_notification' => 'boolean'
        ]);

        $technician = User::findOrFail($validated['technician_id']);

        if ($technician->role !== 'technician') {
            return response()->json(['message' => 'Sadece teknik servis personeli atanabilir.'], 422);
        }

        DB::transaction(function () use ($serviceRecord, $technician, $validated, $request) {
            $serviceRecord->update(['technician_id' => $technician->id]);

            if ($validated['send_notification'] ?? true) {
                Mail::to($technician->email)->send(
                    new TechnicianAssigned($serviceRecord, $technician, $request->user())
                );
            }
        });

        return response()->json([
            'data' => $serviceRecord->fresh()
        ]);
    }

    /**
     * Display the timeline of the specified service record.
     *
     * @param  \App\Models\ServiceRecord  $serviceRecord
     * @return \Illuminate\Http\JsonResponse
     */
    public function timeline(ServiceRecord $serviceRecord)
    {
        // Müşteri sadece kendi kaydının zaman çizelgesini görebilir
        if (request()->user()->role === 'customer' && $serviceRecord->customer_email !== request()->user()->email) {
            return response()->json(['message' => 'Bu kayda erişim yetkiniz yok.'], 403);
        }

        // Teknisyen sadece kendisine atanan kaydın zaman çizelgesini görebilir
        if (request()->user()->role === 'technician' && $serviceRecord->technician_id !== request()->user()->id) {
            return response()->json(['message' => 'Bu kayda erişim yetkiniz yok.'], 403);
        }

        return response()->json([
            'data' => $serviceRecord->statusHistory()->with('changedBy')->get()
        ]);
    }

    /**
     * Servis kayıtları için rapor oluşturur
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function report(Request $request)
    {
        $query = ServiceRecord::query();

        // Müşteri sadece kendi kayıtlarını görebilir
        if ($request->user()->role === 'customer') {
            $query->where('customer_id', $request->user()->id);
        }

        // Teknisyen sadece kendisine atanan kayıtları görebilir
        if ($request->user()->role === 'technician') {
            $query->where('technician_id', $request->user()->id);
        }

        // Tarih aralığı filtresi
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        // Tüm kayıtları al
        $records = $query->get();
        $total = $records->count();

        // Müşteri bazında raporlama
        $byCustomer = $records->groupBy('customer_id')
            ->map(function ($items) use ($total) {
                $count = $items->count();
                return [
                    'customer_name' => $items->first()->customer->name,
                    'count' => $count,
                    'percentage' => round(($count / $total) * 100, 2)
                ];
            })->values();

        return response()->json([
            'success' => true,
            'data' => [
                'records' => $records,
                'total' => $total,
                'by_customer' => $byCustomer
            ]
        ]);
    }

    /**
     * Generate a unique tracking number.
     *
     * @return string
     */
    protected function generateTrackingNumber()
    {
        do {
            $trackingNumber = 'TS' . strtoupper(Str::random(8));
        } while (ServiceRecord::where('tracking_number', $trackingNumber)->exists());

        return $trackingNumber;
    }

    public function destroy(ServiceRecord $serviceRecord)
    {
        $serviceRecord->delete();

        return response()->json(null, 204);
    }

    public function statusDistributionStatistics(Request $request)
    {
        $this->authorize('viewAny', ServiceRecord::class);

        $query = ServiceRecord::query();

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        $total = $query->count();
        $distribution = $query->select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get()
            ->map(function ($item) use ($total) {
                return [
                    'status' => $item->status,
                    'count' => $item->count,
                    'percentage' => round(($item->count / $total) * 100, 2)
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $distribution
        ]);
    }

    public function technicianPerformanceStatistics(Request $request)
    {
        $this->authorize('viewAny', ServiceRecord::class);

        $query = ServiceRecord::query()
            ->whereNotNull('technician_id')
            ->with('technician:id,name');

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        $performance = $query->select('technician_id', DB::raw('
                count(*) as total_records,
                sum(case when status = "completed" then 1 else 0 end) as completed_records,
                avg((strftime("%s", updated_at) - strftime("%s", created_at)) / 3600.0) as average_completion_time
            '))
            ->groupBy('technician_id')
            ->get()
            ->map(function ($item) {
                return [
                    'technician_id' => $item->technician_id,
                    'technician_name' => $item->technician->name,
                    'total_records' => $item->total_records,
                    'completed_records' => $item->completed_records,
                    'average_completion_time' => round($item->average_completion_time, 2),
                    'success_rate' => round(($item->completed_records / $item->total_records) * 100, 2)
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $performance
        ]);
    }

    public function completionTimeStatistics(Request $request)
    {
        $this->authorize('viewAny', ServiceRecord::class);

        $query = ServiceRecord::query()
            ->where('status', 'completed');

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        $completionTimes = $query->select(DB::raw('
                avg((strftime("%s", updated_at) - strftime("%s", created_at)) / 3600.0) as average,
                min((strftime("%s", updated_at) - strftime("%s", created_at)) / 3600.0) as min,
                max((strftime("%s", updated_at) - strftime("%s", created_at)) / 3600.0) as max
            '))
            ->first();

        return response()->json([
            'success' => true,
            'data' => [
                'average_completion_time' => round($completionTimes->average, 2),
                'min_completion_time' => round($completionTimes->min, 2),
                'max_completion_time' => round($completionTimes->max, 2)
            ]
        ]);
    }

    public function monthlyStatistics(Request $request)
    {
        $this->authorize('viewAny', ServiceRecord::class);

        $query = ServiceRecord::query();

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        $statistics = $query->select(
                DB::raw('strftime("%Y-%m", created_at) as month'),
                DB::raw('count(*) as total_records'),
                DB::raw('sum(case when status = "completed" then 1 else 0 end) as completed_records'),
                DB::raw('sum(case when status = "cancelled" then 1 else 0 end) as cancelled_records')
            )
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->map(function ($item) {
                return [
                    'month' => $item->month,
                    'total_records' => $item->total_records,
                    'completed_records' => $item->completed_records,
                    'cancelled_records' => $item->cancelled_records,
                    'completion_rate' => round(($item->completed_records / $item->total_records) * 100, 2)
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $statistics
        ]);
    }

    public function customerStatistics(Request $request)
    {
        $this->authorize('viewAny', ServiceRecord::class);

        $query = ServiceRecord::query()
            ->with('customer:id,name');

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        $statistics = $query->select(
                'customer_id',
                DB::raw('count(*) as total_records'),
                DB::raw('sum(case when status = "completed" then 1 else 0 end) as completed_records'),
                DB::raw('avg((strftime("%s", updated_at) - strftime("%s", created_at)) / 3600.0) as average_completion_time')
            )
            ->groupBy('customer_id')
            ->get()
            ->map(function ($item) {
                return [
                    'customer_id' => $item->customer_id,
                    'customer_name' => $item->customer->name,
                    'total_records' => $item->total_records,
                    'completed_records' => $item->completed_records,
                    'average_completion_time' => round($item->average_completion_time, 2)
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $statistics
        ]);
    }
}
