<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CustomerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Customer::query();

        // Filtreleme
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Aktif/Pasif filtreleme
        if ($request->has('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        // Sıralama
        $sort = $request->input('sort', 'created_at');
        $direction = $request->input('direction', 'desc');
        $query->orderBy($sort, $direction);

        return response()->json([
            'success' => true,
            'data' => $query->paginate($request->input('per_page', 15))
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'type' => 'required|in:individual,corporate',
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:customers',
            'phone' => 'required|string|unique:customers',
            'address' => 'nullable|string',
            'tax_number' => 'nullable|string|required_if:type,corporate',
            'tax_office' => 'nullable|string|required_if:type,corporate',
            'notes' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $customer = Customer::create($request->all());

        return response()->json([
            'success' => true,
            'data' => $customer,
            'message' => 'Müşteri başarıyla oluşturuldu.'
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Customer $customer)
    {
        $customer->load(['serviceRecords' => function($query) {
            $query->latest()->take(5);
        }]);

        return response()->json([
            'success' => true,
            'data' => $customer
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Customer $customer)
    {
        $validator = Validator::make($request->all(), [
            'type' => 'sometimes|required|in:individual,corporate',
            'name' => 'sometimes|required|string|max:255',
            'email' => 'nullable|email|unique:customers,email,' . $customer->id,
            'phone' => 'sometimes|required|string|unique:customers,phone,' . $customer->id,
            'address' => 'nullable|string',
            'tax_number' => 'nullable|string|required_if:type,corporate',
            'tax_office' => 'nullable|string|required_if:type,corporate',
            'notes' => 'nullable|string',
            'is_active' => 'sometimes|required|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $customer->update($request->all());

        return response()->json([
            'success' => true,
            'data' => $customer,
            'message' => 'Müşteri başarıyla güncellendi.'
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Customer $customer)
    {
        $customer->delete();

        return response()->json([
            'success' => true,
            'message' => 'Müşteri başarıyla silindi.'
        ]);
    }

    public function serviceHistory(Customer $customer)
    {
        $serviceRecords = $customer->serviceRecords()
            ->with(['items', 'statuses'])
            ->latest()
            ->paginate(10);

        return response()->json([
            'success' => true,
            'data' => $serviceRecords
        ]);
    }

    public function statistics(Customer $customer)
    {
        $stats = [
            'total_services' => $customer->serviceRecords()->count(),
            'active_services' => $customer->serviceRecords()
                ->whereNotIn('status', ['completed', 'delivered', 'cancelled'])
                ->count(),
            'total_spent' => $customer->serviceRecords()->sum('total_cost'),
            'warranty_services' => $customer->serviceRecords()
                ->where('is_warranty', true)
                ->count()
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }
}
